#include <iostream>
#include <string>
#include <vector>
#include <iomanip>
#include <cmath>
#include <fstream>
#include <cstdio>
using namespace std;

struct Point
{
    double x, y;

    Point(double x, double y) :
        x(x),
        y(y)
    {
    }

    Point() :
        x(0.0),
        y(0.0)
    {
    }

};

ostream& operator << (ostream& output_stream, const Point& point)
{
    output_stream << point.x << " " << point.y << endl;
    return output_stream;
}

void Normalize(Point& point)
{
    double length = sqrt(point.x * point.x + point.y * point.y);

    point.x /= length;
    point.y /= length;
}

Point operator + (const Point& lhs, const Point& rhs)
{
    return Point(lhs.x + rhs.x, lhs.y + rhs.y);
}

Point operator * (const double d, const Point& point)
{
    return Point(point.x * d, point.y * d);
}

struct Line
{
    double A, B, C;

    Line(const Point& lhs, const Point& rhs)
    {
        A = lhs.y - rhs.y;
        B = rhs.x - lhs.x;
        C = -(lhs.x * A + lhs.y * B);
    }
};

void Normalize(Line& line)
{
    double length = sqrt(line.A * line.A + line.B * line.B);

    line.A /= length;
    line.B /= length;
    line.C /= length;
}

Point Intersection(const Line& line_first, const Line& line_second)
{
    double delta = line_first.A * line_second.B - line_first.B * line_second.A;

    double delta_x = (-line_first.C) * line_second.B  - line_first.B * (-line_second.C);

    double delta_y = line_first.A * (-line_second.C) - (-line_first.C) * line_second.A;

    return Point(delta_x / delta, delta_y / delta);
}

Point GetOrthogonal(const Line& line)
{
    return Point(line.A, line.B);
}

double DistToLine(Line& line, const Point& point)
{
    Normalize(line);

    return fabs(line.A * point.x + line.B * point.y + line.C);
}

Point GetSymmetric(Line& line, const Point& point)
{
    double dist = DistToLine(line, point);

    Point ort = GetOrthogonal(line);

    Normalize(ort);

    Point point_first = point + (2.0 * dist) * ort;
    Point point_second = point + (-2.0 * dist) * ort;

    double dist_first = DistToLine(line, point_first);
    double dist_second = DistToLine(line, point_second);

    if (dist_first < dist_second) {
        return point_first;
    } else {
        return point_second;
    }
}

struct Data
{
    Line* line;
    Point* point;

    bool is_line;
};

Data MergeData(const Data& lhs, const Data& rhs)
{
    Data result;

    if (!lhs.is_line && !rhs.is_line) {
        result.line = new Line(*lhs.point, *rhs.point);
        result.is_line = true;
    } else if (!lhs.is_line && rhs.is_line) {
        result.point = new Point(GetSymmetric(*rhs.line, *lhs.point));
        result.is_line = false;
    } else if (lhs.is_line && !rhs.is_line) {
        result.point = new Point(GetSymmetric(*lhs.line, *rhs.point));
        result.is_line = false;
    } else {
        result.point = new Point(Intersection(*lhs.line, *rhs.line));
        result.is_line = false;
    }

    return result;
}

double GetNumber(const string& str, int& pos)
{
    int result = 0;

    int sign = 1;

    if (str[pos] == '-') {
        sign = -1;
        ++pos;
    }

    while (pos < (int) str.length() && isdigit(str[pos])) {
        result = result * 10 + (str[pos] - '0');
        ++pos;
    }

    return result * sign;
}

struct Vertex
{
    Data data;
    bool is_query;

    vector<Vertex*> childs;
};

Vertex* Build(const string& str, int& pos)
{
    Vertex* result = new Vertex();

    result->is_query = true;

    while (pos < (int) str.length()) {
        if (str[pos] == '(') {
            ++pos;
            result->childs.push_back(Build(str, pos));
        } else if (str[pos] == ')') {
            ++pos;
            break;
        } else if (str[pos] == '-' || isdigit(str[pos])) {
            result->is_query = false;

            double num_first = GetNumber(str, pos);

            ++pos;

            double num_second = GetNumber(str, pos);

            result->data.is_line = false;
            result->data.point = new Point(num_first, num_second);
        } else {
            ++pos;
        }
    }

    return result;
}

Data CalcVertex(Vertex* vertex)
{
    if (vertex->is_query) {
        Data result = CalcVertex(vertex->childs[0]);
        for (int i = 1; i < (int) vertex->childs.size(); ++i) {
            Data current = CalcVertex(vertex->childs[i]);

            result = MergeData(result, current);
        }
        return result;
    } else {
        return vertex->data;
    }
}

void ShowDepth(int depth)
{
    for (int i = 0; i < depth; ++i) {
        cout << " ";
    }
}

void ShowTree(Vertex* vertex, int depth)
{
    ShowDepth(depth);
    if (vertex->is_query) {
        cout << "@" << endl;
        for (int i = 0; i < (int) vertex->childs.size(); ++i) {
            ShowTree(vertex->childs[i], depth + 1);
        }
    } else {
        cout << (*vertex->data.point) << endl;
    }
}

int main()
{
  //  freopen("input.txt", "r", stdin);

    cout << fixed << setprecision(15);

    ios_base::sync_with_stdio(false);

    while (true) {
        string str;

        cin >> str;

        if (str == "#") {
            break;
        }

        int temp = 0;
        Vertex* tree = Build(str, temp);

       // ShowTree(tree, 0);

        Data data = CalcVertex(tree);

        cout << (*data.point);
    }
    return 0;
}
